name	floppy


;
;
;	floppy disk driver for the VICKI portable
;
;


;
;	errors generated by the floppy device driver . . .
;
;
no_sync		equ	01h;	no sync	
no_header_id	equ	02h;	header id didn't follow sync
header_checksum	equ	03h;	header checksum error
wrong_track	equ	04h;	requested track <> header track
wrong_sector	equ	05h;	no match on sector - error count expired
no_data_id	equ	06h;	data id didn't follow header
data_checksum	equ	07h;	data checksum error
long_sync	equ	08h;	sync present too long
door_opened	equ	09h;	door opened during operation
gcr_error	equ	0Ah;	gcr error
bad_dun		equ	20h;	invalid device/unit field in lrb
bad_da		equ	21h;	invalid sector address in lrb
no_system	equ	99h;	floppy label indicates no O/S on disk



page
;
;	define I/O registers for speed control (motor clock)
;
motor_time	segment at 0E000h;	8253 address space


		org	025h;		drive 0's timer
drive_0_time	equ	$;

		org	026h;		drive 1's timer
drive_1_time	equ	$;

		org	027h;		control byte
timer_control	equ	$;
drive_0_timer	equ	076h;		says set timer for drive 0
drive_1_timer	equ	0B6h;		says set timer for drive 1
drive_0_off	equ	072h;		says set motor off for drive 0
drive_1_off	equ	0B2h;		says set motor off for drive 1

motor_time	ends;



	
;
;	define I/O registers for floppy disk
;
ioports		segment at 0E800h;	6500 address space

;
;	stepper motor control
;
		org	0A0h;		"peripheral B port"
motor_1		equ	byte ptr $;
motor_ready	equ	1;		flags have a motor ready indicator
door_sense	equ	2;		flags have a door sense
full_power	equ	4;		flags is a full-powered system
must_be_0	equ	8;		must be zero if drives present
f_stepbits	equ	0F0h;		stepper phase mask
f_stptk0	equ	0A0h;		flags if on track 0,4,8,...

		org	0A1h;		"peripheral A port"
motor_0		equ	byte ptr $;	same as above for stepper
no_drive_0	equ	1;		NOT drive A present
no_drive_1	equ	2;		NOT drive B present
double_sided	equ	4;		NOT double sided
track0_sense	equ	8;		flags have a track 0 sense

		org	0A2h;		"data direction register B"
motor_ddb	equ	byte ptr $;

		org	0A3h;		"data direction register A"
motor_dda	equ	byte ptr $;	same as above

page
;
;	miscellaneous control registers
;
		org	0ACh;		"peripheral control register"
mode		equ	byte ptr $;	selects data mode

		org	0AEh;		"interrupt enable register"
intps		equ	byte ptr $;	interrupt enable flags

		org	0C0h;		"peripheral B port"
read_perb	equ	byte ptr $;	floppy door indicator
f_rdy0		equ	01h;		output motor speed for A
f_rdy1		equ	02h;		output motor speed for B
spc_data_rdy	equ	01h;		output controller handshake data ready
spc_data_rcvd	equ	02h;		output controller handshake data taken
f_screset	equ	04h;		output speed controller reset
f_door1		equ	08h;		input door open sense for B
f_door0		equ	10h;		input door open sense for A
f_s0enab	equ	40h;		NOT output stepper enable drive A
f_s1enab	equ	80h;		NOT output stepper enable drive B

		org	0C1h;		"peripheral A port"
clrgcr		equ	byte ptr $;	if read, will clear the gcr
	
		org	0C2h;		"data direction register, port B"
data_ddb	equ	byte ptr $;

		org	0C3h;		"data direction register, port A"
data_dda	equ	byte ptr $;

		org	0CCh;		"peripheral control register"
readwrite_erase	equ	byte ptr $;	disk read/write and erase head

		org	0CDh;		"interrupt flag register"
fgcr		equ	byte ptr $;	gcr register
gcrerr		equ	02h;		gcr error bit


page
		org	0CFh;		same as 0C1, but doesn't reset if read
read_pera	equ	byte ptr $;	sync indicator
f_led0		equ	01h;		output LED for drive A
drv0_trk0	equ	02h;		NOT drive A track 0 sense
f_led1		equ	04h;		output LED for drive B
drv1_trk0	equ	08h;		NOT drive B track 0 sense
f_sidesel	equ	10h;		output disk side selection
f_selbit	equ	20h;		output select drive A or B
f_wpsbit	equ	40h;		output write protect sense
fsynbit		equ	80h;		sync detect bit


;
;	registers for reading/writing disk data
;
		org	0E0h;		"data direction register port B"
dskout		equ	byte ptr $;	disk data outputs

		org	0E1h;		"data direction register port A"
dskdata		equ	byte ptr $;	disk data

		org	0E2h;		"data direction register port B"
diskout_ddb	equ	byte ptr $;

		org	0E3h;		"data direction register port A"
diskin_dda	equ	byte ptr $;

		org	0EBh;		"auxilliary control register"
acr		equ	byte ptr $;


ioports	ends;



page
;
;	miscellaneous disk equates
;

trk_mask	equ	7fh;		remove side information, get track only
speed_zones	equ	9;		there are 9 speed zones on a disk
max_tracks	equ	104;		maximum tracks arm may have to travel
label_size	equ	80h;		size of the floppy disk label
os_label	equ	0FFh;		flags system disk label
fhid		equ	7;		value for header id on disk
fdid		equ	8;		value for data id on disk




page

data	segment public 'data'

	extrn	sec:byte;		working sector
	extrn	trk:byte;		working track
	extrn	curspd:word;		speed zone we're currently at
	extrn	fldma:dword;		dma address in variables
	extrn	state:byte;		flags to determine if a disk there
	extrn	curtrk:byte;		track we're currently at (each drive)
	extrn	seccnt:byte;		count down of number of sectors to read
	extrn	sav_seccnt:byte;	permanent copy of number sectors read
	extrn	sector_retry:byte;	retry count for reading bad sectors

include	bt1lrb.str;
;
;	FORMAT OF THE LOAD REQUEST BLOCK (LRB)
;
lrbs	struc;

op	dw	(?);		Operation Code
dun	dw	(?);		Device/Unit Number
da	dw	(?);		Physical Address on Volume
	dw	(?);
dma	dw	(?);		Direct Memory Address
	dw	(?);
blkcnt	dw	(?);		Number of Blocks in Transfer
status	db	(?);		STATUS code returned 
ssz	dw	(?);		Sector Size 
loadaddr dw	(?);		Segment to Load into - 0 => Load High
loadpara dw	(?);		Paragraph Count 
loadentry dw	2 dup (?);	Entry Point - Seg=0 => Use "loadaddr"
lrbs	ends;

	extrn	lrb:byte;		load request block (I/O parameters)

include	bt1fdlbl.str;
;
;	FLOPPY DISK LABEL FORMAT
;

fd_lbl		struc;

valid		dw	?;		must be ff00 to be a loadable O/S
load		dw	?;		system load address
paras		dw	?;		size of system
entry_offset	dw	?;		program's start offset from segment
entry_segment	dw	?;		program's start segment
disk_id		db	8 dup (?);	disk identification (unused)
system_id	db	8 dup (?);	system identification (unused)
sector_size	dw	?;		floppy's sector size
data_start	dw	?;		offset to unreserved region
boot_start	dw	?;		boot sector origin
flags		db	?;		(unused)
int_fact	db	?;		sector interleave (unused)
disk_type	db	?;		disk version
reserved	db	3 dup (?);	(unused)
speed_control	db	18 dup (?);	speed control table
zone_table	db	15 dup (?);	locations of speed change
sectors_track	db	15 dup (?);	sectors per track

fd_lbl		ends;

	extrn	bootr:byte;		floppy disk label buffer

data	ends



cgroup	group code;
dgroup	group data;
assume	cs:cgroup,ds:dgroup,es:ioports;




code	segment public 'code';

extrn	err_intern:near;
extrn	time:near;
public	fd_reset;
public	fd_cup;
public	fd_dvcrdy;
public	fd_online;
public	fd_read;
public	fd_quiesce;
public	fd_status;


page
;
;	data area for tables and constants
;


;
;	speed tables concerning the description of VICKI drives
;

speed_for_zones		dw	3965,3741,3536,3331,3126,2921,2716,2493,2400
last_track_for_zone	db	3,15,26,37,47,59,70,82,97
sectors_per_track	db	19,18,17,16,15,14,13,12,11




page
;
;	reset function for floppy disk
;
;	input:	none
;
;	output:	all registers destroyed
;
fd_reset	proc;

	xor	ax,ax;			flag good result
	mov	word ptr state,ax;	don't know if either drive has a disk
	mov	curspd,ax;		and flag we're on speed zone 0

	ret;

fd_reset	endp;



;
;	checks to see if floppy disk control unit is present
;
;	input:	parameters as specified in load request block
;
;	output:	z-flag is set if and only if operation is successful
;		lrb's status field is set to which drives are present
;		all registers are destroyed
;

fd_cup	proc;

	mov	ax,seg ioports;		address the floppy I/O registers
	mov	es,ax;

;
;	check configuration bytes to determine if controller is present
;

	test	es:[motor_1],must_be_0;	is controller present
	jnz	cu_not_there;		no

	xor	dl,dl;			preset to no drives present
	test	es:[motor_0],no_drive_0;is drive 0 there ?
	jnz	no_left_drive;		no, skip
	inc	dl;			flag drive 0 present

no_left_drive:;
	test	es:[motor_0],no_drive_1;is drive 1 there ?
	jnz	no_right_drive;		no, skip
	add	dl,2;			flag drive 1 present

no_right_drive:;
	or	dl,dl;			any drives ?
	jz	cu_not_there;		no, don't bother with floppies

	mov	lrb.status,dl;		indicate which drives are there


;
;	controller is present, initialize it
;

	call	resethdw;		clear all floppy I/O registers

;
;	initialize direction of lines
;
	mov	es:[data_dda],f_led0 or f_led1 or f_selbit or f_sidesel;
	mov	es:[data_ddb],f_screset+f_s0enab+f_s1enab;

;
;	erase head off, write off
;
	mov	es:[readwrite_erase],0EEh;

;
;	set registers for disk data
;
	mov	byte ptr es:[diskout_ddb],0FFh;	direction for disk output
	mov	byte ptr es:[acr],01h;	enable latching on diskdata (input)


;
;	return, control unit is there
;
	xor	ax,ax;			flag control unit is there

cu_not_there:;
	ret;				with z-flag being the result


fd_cup	endp;





page
;
;	test if floppy drive is ready (that is, a disk is present)
;	(also serves as status since motor is always turned off)
;
;	inputs:		load request block set up
;
;	results:	z-flag is set if device is ready
;			all registers are destroyed
;
fd_dvcrdy	proc;
fd_status	proc;

	mov	ax,seg ioports;		address the floppy I/O registers
	mov	es,ax;

	mov	bx,lrb.dun;		get device and unit
	and	bx,1;			get drive number (0 or 1)

	mov	si,1;			only try once to see if disk already in

;
;	if there is a door sense, use that for device ready
;	(otherwise, it's more complicated to determine is a disk is there)
;
ready_retry:;
	test	es:[motor_1],door_sense;have a door ?
	jz	use_index;		no, must use the index hole

check_again:;
	mov	cl,es:[read_perb];	read door statuses

	mov	dl,f_door0;
	or	bl,bl;			which drive ?
	jz	want_door0;		zero, skip
	mov	dl,f_door1;		select which drive's door

want_door0:;
	and	cl,dl;			cl has result for now

	mov	ax,200;			wait for 20,000 microseconds	
	call	time;			for debouncing the door status

	mov	al,byte ptr es:[read_perb];	read it again
	and	al,dl;			al has status for this time

	cmp	al,cl;			bouncing ?
	jnz	check_again;		yes, try again to debounce result

	or	al,al;			set z-flag to the result
	ret;				(zero if it's present)


;
;	use index hole and motor if we don't know the state of
;	the disk (otherwise look for changes in index and write protect)
;
use_index:;
	call	sel_drv;		select the drive

	mov	dl,f_door0;
	or	bl,bl;			which drive ?
	jz	do_want_door0;		zero, skip
	mov	dl,f_door1;		select which drive's door

do_want_door0:;
	cmp	state[bx],0;		do we know if there is a disk ?
	jnz	test_change;		yes, see if there is a change

;
;	We don't know if there is a disk inserted,
;	we must turn the motor on to check the index hole.
;	To tell the difference between a disk being pulled out
;	and a true index sense, look for hole/no hole/hole sequence.
;	Look for a disk and door closure for up to 5 seconds.
;

	xor	ax,ax;			select speed zone 0
	call	motor_speed;		and turn on the motor

	mov	bp,1;			preset to flag no disk present

wait_for_disk:;
	xor	dh,dh;			look for a change from 0 to 1
	call	index_change;		(ie, no hole to hole)
	jz	no_disk;		no change occurred, no disk

	mov	dh,ah;			save present index value
	call	index_change;		to go from hole to no hole
	jz	no_disk;		no change occurred, no disk

	mov	dh,ah;			save present index value
	call	index_change;		to go from no hole to hole
	jz	no_disk;		no change occurred, no disk

	xor	bp,bp;			flag there's a disk inserted
	jmp short got_disk;

;
;	no disk is inserted yet, retry if required
;
no_disk:;
	dec	si;			decrement retry counter
	jnz	wait_for_disk;		and try again

;
;	wait until motor stops to get index's sense
;
got_disk:;
	call	motor_off;		turn the motor back off

wait_for_stable:;
	mov	dh,es:[read_perb];	get current index state
	and	dh,dl;			look only at this drive's bit
	
	call	index_change;		see if it changes
	jnz	wait_for_stable;	it did, try it again

	mov	al,es:[read_pera];	get write protect status
	and	al,f_wpsbit;
	or	al,ah;			combine with index status
	or	al,080h;		and a flag saying we know about the disk
	mov	state[bx],al;		store the drive's state

	mov	ax,bp;
	or	state[bx],al;		flag whether disk is present

	or	bp,bp;			also flag in z-flag
	ret;

;
;	we know the state of the drive (if there's a disk in it)
;	so we need only turn the motor on if there are transitions
;
test_change:;
	mov	ah,es:[read_perb];	get new index state
	and	ah,dl;			look only at this drive's bit
	mov	al,es:[read_pera];	get write protect status
	and	al,f_wpsbit;
	or	al,ah;			combine with index status

	mov	ah,state[bx];		get old state information
	and	ah,07Eh;		ignore "know" and "status" bits

	cmp	ah,al;			did it change ?
	jz	same_status;		no, return same old status

	mov	state[bx],0;		yes, flag do not know state

	mov	si,10;			minimum of 0.5 seconds per sense,
;					number of retries in 5 seconds
	jmp	ready_retry;		and use index to determine state

same_status:;
	test	state[bx],1;		disk present ?
	ret;				return result in z-flag


fd_status	endp;
fd_dvcrdy	endp;




;
;	Bring a floppy disk drive on-line.
;
;	(Read the label, update the boot vector table,
;	load request block, and set the speed control
;	processor.  The drive will have the motor on at
;	track 0 with the new speed set.  On error, the
;	motor is off, speed control processor isn't loaded.)
;
;	inputs:		load request block set up
;
;	outputs:	lrb updated, z-flag is set if success
;			all registers are destroyed
;
fd_online	proc;

	mov	ax,seg ioports;		address the floppy I/O registers
	mov	es,ax;

;
;	validate device/unit parameter from lrb
;
	mov	bx,lrb.dun;		get the device and unit

	cmp	bx,1;			drive too big ?
	jbe	no_error_in_dun;	no, skip the error

	mov	lrb.status,bad_dun;	set error of invalid dun
	jmp	online_error;

no_error_in_dun:;


;
;	get the drive into a known state (on track 0)
;	(if we can read a header, we know a disk is inserted)
;

	call	sel_drv;		select the drive

	lea	si,ioports:motor_0;	base of speed control registers
	sub	si,bx;			select drive's register
	mov	byte ptr es:[si],f_stptk0;	set stepper to track 0,4,8,...
	mov	byte ptr es:[si+2],f_stepbits;	set data direction register

;
;	step outwards until on track zero
;
	call	zero_chk;		on track zero ?
	jz	set_on_track_zero;	yes, quit

	call	step_in;
	call	step_in;		first, step in 4 tracks
	call	step_in;		(since we may be on track -4)
	call	step_in;

	mov	bp,max_tracks/4;	could take up to 104 tracks

step_again:;
	call	zero_chk;		on track zero ?
	jz	set_on_track_zero;	yes, quit

	call	step_out;		no, step one more track
	call	step_out;
	call	step_out;		(keep in multiple of four
	call	step_out;		so stepper is still on phase 0)

	dec	bp;			one more try ?
	jnz	step_again;		yes, step some more

	mov	lrb.status,wrong_track;	set error of bad track
	jmp	online_error;

;
;	on track 0 and a disk is present
;
set_on_track_zero:;
	mov	byte ptr curtrk[bx],0;	say on track 0 now


;
;	read label into buffer
;

	call	sel_drv;		select the drive

	mov	ax,label_size;		set size of label
	mov	lrb.ssz,ax;

	xor	ax,ax;	
	mov	lrb.da,ax;		disk address of zero
	mov	lrb.da+2,ax;

	mov	word ptr lrb.dma+2,ax;	segment of label buffer
	lea	di,dgroup:bootr;	offset of label buffer
	mov	lrb.dma,di;

	inc	ax;
	mov	lrb.blkcnt,ax;		read one block

	call	fd_read;		read the label
	jz	read_ok;
	jmp	online_error;		couldn't read, error

read_ok:;
	cmp	bootr.valid,os_label;	valid label ?
	jz	label_valid;		yes, skip

	mov	lrb.status,no_system;	error, label says no O/S
	jmp	online_error;

label_valid:;

;
;	update lrb from floppy disk's label
;
	mov	ax,bootr.sector_size;
	mov	lrb.ssz,ax;			store sector size

	mov	ax,bootr.load;			copy load address for image
	mov	lrb.loadaddr,ax;

	mov	ax,bootr.paras;			and size in paragraphs
	mov	lrb.loadpara,ax;

	mov	ax,bootr.entry_offset;
	mov	word ptr lrb.loadentry,ax;	copy the O/S's entry address
	mov	ax,bootr.entry_segment;
	mov	word ptr lrb.loadentry+2,ax;

	mov	ax,bootr.boot_start;		and the disk address
	mov	lrb.da,ax;

	xor	ax,ax;			return success in z-flag
	ret;

;
;	here if error occurred
;
online_error:;
	call	motor_off;		turn off the floppy's motor
any_error:;
	mov	ax,1;			flag an error
	or	ax,ax;			z-flag is nz for errors
	ret;


fd_online	endp;




;
;	read a boot image from the floppy disk
;
;	inputs:		load request block set up
;
;	outputs:	z-flag is set for successful read
;			lrb updated
;			all registers destroyed
;
fd_read		proc;

	mov	ax,seg ioports;		address the floppy I/O registers
	mov	es,ax;

;
;	check validity of requested operation
;
	xor	cl,cl;			initially, no error found

	mov	bx,lrb.dun;		get device and unit
	cmp	bx,1;			more than 1 ?
	jbe	in_range;		no, skip

	mov	cl,bad_dun;		invalid unit specified

in_range:;
	cmp	word ptr lrb.da+2,0;	high-order sector address
	jz	sector_ok;		zero, valid

	mov	cl,bad_da;		invalid sector address

sector_ok:;
	or	cl,cl;			errors ?
	jz	no_errors;		no, skip

	mov	lrb.status,cl;		set the return status
	jmp	any_error;

;
;	set up for the read
;
no_errors:;
	mov	ax,word ptr lrb.dma;
	mov	cx,word ptr lrb.dma+2;
	mov	word ptr fldma,ax;	save dma address
	mov	word ptr fldma+2,cx;

	call	sel_drv;		select the drive


;
;	loop to read each block from disk, finally filling the request
;

read_next_block:;

	cmp	lrb.blkcnt,0;		done ?	
	jnz	read_more;		no
	jmp	dquit;			yes.

read_more:;
	mov	dx,word ptr lrb.da;	get absolute disk address

;
;	find out what zone the particular sector is in
;	in order to compute what track and sector to access
;	(non-trivial since there's a variable number of sectors per track)
;
	xor	bp,bp;			index through each speed zone
	xor	cl,cl;			initialize starting track of this zone
zone_search:;
	mov	al,cs:last_track_for_zone[bp];	get ending track of this zone
	inc	al;			now, it's starting track of next zone
	sub	al,cl;			minus starting track of this zone
;					yields size of this zone in tracks
	xor	ah,ah;			prepare for multiplication
	mul	cs:sectors_per_track[bp];tracks in this zone * sectors per track
;					yields sectors in this zone
	cmp	dx,ax;			sector looking for in this zone ?
	jb	end_search;		yes, found it.

	sub	dx,ax;			subtract sectors in this zone
;					from sector looking for
	mov	cl,cs:last_track_for_zone[bp];	last track of this zone
	inc	cl;			now, it's starting track of next zone

	inc	bp;			try next zone
	jmp	zone_search;

;
;	know zone for sector, compute track and sector
;
end_search:;
	mov	ax,dx;
	div	cs:sectors_per_track[bp];divide sector displacement in zone
;					by sectors per track . . .
;					ah = sector offset in track (modulo)
;					al = track offset in zone (quotient)
	mov	sec,ah;			save the sector
	add	al,cl;			add first track in zone to track
;					offset in zone (is track to access)
	mov	trk,al;			and save the track

	mov	dl,cs:sectors_per_track[bp];	get number of sectors per track
	sub	dl,sec;			sectors per track minus
;					sector we're looking for
;					yields sectors left in track

	xor	dh,dh;
	cmp	dx,lrb.blkcnt;		sectors left less than number requested?
	jbe	skip;			yes, can only read rest of track
	mov	dx,lrb.blkcnt;		no, can read entire number requested
skip:;
	mov	seccnt,dl;		save number of sectors we're reading
	mov	sav_seccnt,dl;		and permanently save it

page
;
;	perform the read (up to a track's worth)
;

	mov	bp,10;			retries for offtrack seeks
;
;	seek to the desired track
;

;
;	find index and set speed
;
find_track:;
	lea	si,ioports:motor_0;	base of speed control registers
	sub	si,bx;			select drive's register

	mov	al,trk;			track we want
	call	get_zon;		get zone number for this track

	call	motor_speed;		and put the motor at that speed

;
;	now, seek the the right track
;
	mov	dl,curtrk[bx];		get our present track setting
move_more:;
	cmp	trk,dl;			compare to where we want to be
	je	just_settle;		we're there, just let it settle

	jb	step_it_out;		outwards
	call	step_in;		inwards
	inc	dl;			new track number
	jmp	move_more;

step_it_out:;
	call	step_out;		otherwise move some more
	dec	dl;			new track number
	jmp	move_more;

;
;	at the new track
;
just_settle:;
	mov	curtrk[bx],dl;		save new track number

	mov	ax,350;			wait 35,000 microseconds for settling
	call	time;			stepper or speed processor settle time

page


;
;	head is on the right track, read the boot image
;


readseek:;
	mov	sector_retry,10;	retry counter for sector reads
	mov	si,255;			allow 255 soft errors per sector read

seek:;
	mov	dx,5000;		sync time out counter

	mov	cl,76h;			fine-tuned timing value

;
;	wait for a sync to align on the track
;
seek10:;
	test	es:[read_pera],fsynbit;	test if sync present
	jz	seek15;			yes

	shl	al,cl;			no, wait 104 us

	dec	dx;			retry counter for seeing a sync
	jnz	seek10;			retry (total 114 us sample interval)

;
;	error, no sync found
;
	mov	ah,no_sync;		error 1, no sync
	jmp	harderr;


;
;	sync has started, wait for end of it
;
seek15:;
	mov	al,es:[dskdata];	wait for data byte
	mov	dx,69;			retry for data (640us is too much sync)

seek16:;
	test	es:[read_pera],fsynbit;	9us loop
	jnz	seek18;			end of sync, in good shape

	dec	dx;			decrement retry counter
	jnz	seek16;			not too long, wait for end

;
;	error, sync is too long
;
	mov	ah,long_sync;		error 8, sync for too long a time
	jmp short softerr;		can retry this error



;
;	got the end of the sync, look for header
;
seek18:;
	cmp	dx,12;			sync long enough to qualify ?
	jbe	seek10;			no, pretend we didn't see a sync

	mov	al,es:[dskdata];	clear the IFR flag by reading data

	wait;				wait for synchronization from disk

	cmp	es:[dskdata],fhid;	is the data a header ID ?
	jz	seek20;			yes, found header

;
;	error, didn't find a header
;
	mov	ah,no_header_id;	didn't find header
	jmp short softerr;		can retry this one


;
;	found the header, check out our position on the track
;
seek20:;
	wait;				wait for disk data
	mov	al,es:[dskdata];	track number
	mov	dh,al;			checksum calculation in dh

	wait;				wait for disk data
	mov	ah,es:[dskdata];	sector number
	add	dh,ah;			add into checksum

	wait;				wait for disk data
	mov	dl,es:[dskdata];	header checksum

	cmp	dh,dl;			compare checksum
	jz	seek25;			ok

;
;	error, bad checksum in the header
;
	mov	ah,header_checksum;	error 3 is bad header checksum
	jmp short softerr;		can retry this one



;
;	read the header, check if it's the right track and sector
;
seek25:;
	cmp	trk,al;			test if right track
	jz	seek30;			correct, skip

;
;	error, requested track doesn't match the one we got
;
	mov	curtrk[bx],al;		say we're on this track, now
	dec	bp;			up to 10 tries for off-track errors
	jz	too_many_offtracks;
	jmp	find_track;		retry it.

too_many_offtracks:;
	mov	ah,wrong_track;		error 4, track mismatch
harderr2:;
	jmp	harderr;		unrecoverable error


seek30:;
	cmp	sec,ah;			test if right sector
	jz	readsector;		yes, read it



;
;	error, requested sector doesn't match the one we got
;
	mov	ah,wrong_sector;	error 5, try again to get sector

;
;	retryable error
;
softerr:;
	xor	dx,dx;			restore data segment
	mov	ds,dx;

	dec	si;			try again ?
	je	harderr2;		too many errors, harderror
	jmp	seek;			try again



page
;
;	have right sector, time to read it
;
readsector:;
	mov	cx,lrb.ssz;		sector byte count
	lds	di,fldma;		set destination of the read
;					CAUTION:  DS MUST BE RESTORED !!!

	xor	dx,dx;			initialize checksum
	xor	ah,ah;			high byte of read data

doread10:;
	test	es:[read_pera],fsynbit;	wait for data block
	jnz	doread10;

doread11:;
	test	es:[read_pera],fsynbit;	wait for sync to go away
	jz	doread11;		so can reset diskdata latch

	mov	al,es:[dskdata];	reset sync interrupt request
	mov	al,es:[clrgcr];		clear latched gcr

	wait;				wait for disk data
	cmp	es:[dskdata],fdid;	test if a data block ID
	je	doread19;		yes, continue

;
;	error, no data block ID after the header
;
	mov	ah,no_data_id;		error 6, no data ID
	jmp	softerr;		is recoverable


;
;	have a data ID
;
doread19:;
	wait;				wait for disk data
	mov	al,es:[dskdata];	read a byte of data
	add	dx,ax;			calculate the checksum

	mov	ds:[di],al;		save the data byte

	inc	di;			advance "dma" pointer
	loop   doread19;		repeat until a sector is read

	xor	ax,ax;
	mov	ds,ax;			restore data segment

	wait;				wait for disk data
	mov	al,es:[dskdata];	read low byte of the checksum
	wait;				wait for disk data
	mov	ah,es:[dskdata];	read high byte of the checksum

	cmp	ax,dx;			checksum matches ?
	je	okchksum;		yes, continue tests


;
;	error in checksum on reading a sector
;
	mov	ah,data_checksum;	preset to checksum error
	dec	sector_retry;		read error counter
	jz	harderr;		quit, too many retries
	jmp	seek;			start seeking over again


;
;	checksum is valid, now test for gcr error
;
okchksum:;
	test	es:[fgcr],gcrerr;	see if error set
	jz	okgcr;			no, skip

;
;	error, bad gcr
;
	mov	ah,gcr_error;		error code for bad gcr
	jmp	softerr;		but is retryable


page
;
;	the gcr is valid, that sector was successfully read
;
okgcr:;
	inc	sec;			calculate next sector number

	mov	ax,lrb.ssz;		get sector size in bytes
	mov	cl,4;
	shr	ax,cl;			convert to paragraphs
	add	word ptr fldma+2,ax;	bump segment number for dma

	dec	seccnt;			one more sector read on track
	jz	 jobdone;		finished with this track
	jmp	readseek;		do next sector


;
;	error, block was not successfully read
;
harderr:;
	mov	lrb.status,ah;		set return status
	call	motor_off;		turn motor back off
	mov	ah,1;
	or	ax,ax;			reflect error in z-flag
	ret;


;
;	all sectors in the track were read, do rest of the request
;
jobdone:;

	mov	al,sav_seccnt;		number of sectors read
	xor	ah,ah;

	sub	lrb.blkcnt,ax;		decrement blocks to read
	add	lrb.da,ax;		increment disk address by that amount

	jmp	read_next_block;	read next block from disk




;
;	the requested read is completed successfully
;
dquit:;
	mov	ax,word ptr fldma;	get current dma address
	mov	lrb.dma,ax;		and update lrb
	mov	ax,word ptr fldma+2;
	mov	lrb.dma+2,ax;

	call	motor_off;		turn the motor back off
	xor	ax,ax;			flag no error in z-flag
	ret;


fd_read		endp;




page
;
;	quiesces a floppy disk unit (turns the motor off)
;
;	input:	parameters as specified in load request block
;
;	output:	z-flag is set if and only if operation is successful
;		all registers are destroyed
;

fd_quiesce	proc;

	mov	ax,seg ioports;		address the floppy I/O registers
	mov	es,ax;

	mov	bl,byte ptr lrb.dun;	get device and unit
	and	bl,1;			get drive number (0 or 1)

	call	motor_off;		turn off the motor

	xor	ax,ax;			flag no error
	ret;

fd_quiesce	endp;




page

;
;	Step selected drive out one track.
;
;	The stepper motor is issued "tab stops" to position it.
;	A value of "A hex" will position it on track 0,4,8,...
;	A value of "9 hex" will position it on track 3,7,11,...
;	A value of "5 hex" will position it on track 2,6,10,...
;	A value of "6 hex" will position it on track 1,5,9,...
;	By cycling values of A, 9, 5, 6, A, 9, 5, ... the arm will step out.
;
;	input:		bl = drive number
;
;	results:	ax,cx,di,si destroyed
;

step_out_table	db	050h,090h,060h,0A0h;

step_out	proc;

	lea	di,cgroup:step_out_table;address of table to step arm out

step_it:;
	lea	si,ioports:motor_0;	base of speed control registers
	sub	si,bx;			select drive's register
	mov	al,es:[si];		motor information (current position)

	mov	cl,5;
	shr	al,cl;			middle stepper bits are unique index
	and	al,03h;
	xor	ah,ah;
	add	di,ax;			index into stepper transition table

	mov	al,cs:[di];		get new stepper value
	mov	es:[si],al;		and output the new value

	mov	ax,50;			wait 5000 micro seconds
	call	time;			(5 milliseconds)

	ret;

step_out	endp;




;
;	Step selected drive in one track.
;
;	The stepper motor is issued "tab stops" to position it.
;	A value of "A hex" will position it on track 0,4,8,...
;	A value of "6 hex" will position it on track 1,5,9,...
;	A value of "5 hex" will position it on track 2,6,10,...
;	A value of "9 hex" will position it on track 3,7,11,...
;	By cycling values of A, 6, 5, 9, A, 6, 5, ... the arm will step in.
;
;	input:		bl = drive number
;
;	results:	ax,cx,si destroyed
;

step_in_table	db	0A0h,060h,090h,050h;

step_in		proc;

	lea	di,cgroup:step_in_table;address of table to step arm in
	jmp	step_it;		go to common stepping code

step_in		endp;



;
;	tells whether on track 0
;
;	input:		bl = drive to check
;
;	results:	z-flag is set if on track 0
;			ax,cx,si,dx destroyed
;
zero_chk	proc;

	lea	si,ioports:motor_0;	base of speed control registers
	sub	si,bx;			select drive's register
	mov	al,es:[si];		motor information (current position)

	and	al,f_stepbits;		look at step bits
	cmp	al,f_stptk0;		on track 0, 4, 8, etc ?
	jz	on_zero
	jmp	aint_on_zero;		no, can't be on zero

;
;	test configuration byte:
;				TANDEN drives have no track zero sense
;				PHILLIPS drives do
;
on_zero:
	test	es:[motor_0],track0_sense;	have track 0 sense ?
	jz	must_read_header;		no, read track's header

	or	bl,bl;				drive 0 ?
	jnz	test_drv1_trk0;			no, skip

	test	es:[read_pera],drv0_trk0;	on track 0 ?
	ret;					z-flag has the indication

test_drv1_trk0:
	test	es:[read_pera],drv1_trk0;
	ret;


;	we want to read the track's header to see if we're on track zero
;	head is on the track, now ready the motor
;

must_read_header:;
	xor	ax,ax;			select speed zone zero
	call	motor_speed;

	mov	si,5;			retry a soft error 5 times
zero_seek:;
	mov	dx,5000;		sync time out counter

	mov	cl,76h;			fine-tuned timing value

;
;	wait for a sync to align on the track
;
zero_seek10:;
	test	es:[read_pera],fsynbit;	test if sync present
	jz	zero_seek15;		yes

	shl	al,cl;			no, wait 104 us

	dec	dx;			retry counter for seeing a sync
	jnz	zero_seek10;		retry (total 114 us sample interval)

;
;	error, no sync found
;
	jmp	aint_on_zero;		not even a sync, not track zero

;
;	sync has started, wait for end of it
;
zero_seek15:;
	mov	al,es:[dskdata];	wait for data byte
	mov	dx,69;			retry for data (640us is too much sync)

zero_seek16:;
	test	es:[read_pera],fsynbit;	9us loop
	jnz	zero_seek18;		end of sync, in good shape

	dec	dx;			decrement retry counter
	jnz	zero_seek16;		not too long, wait for end

;
;	error, sync is too long
;
	jmp	zero_softerr;		can retry this error



;
;	got the end of the sync, look for header
;
zero_seek18:;
	cmp	dx,12;			sync long enough to qualify ?
	jbe	zero_seek10;		no, pretend we didn't see a sync

	mov	al,es:[dskdata];	reset sync IFR by reading data

	wait;				wait for synchronization from disk

	cmp	es:[dskdata],fhid;	is the data a header ID ?
	jz	zero_seek20;		yes, found header

;
;	error, didn't find a header
;
	jmp	zero_softerr;		can retry this one


;
;	found the header, ensure has a valid checksum
;
zero_seek20:;
	wait;				wait for disk data
	mov	al,es:[dskdata];	track number
	mov	dh,al;			checksum calculation in dh

	wait;				wait for disk data
	mov	ah,es:[dskdata];	sector number
	add	dh,ah;			add into checksum

	wait;				wait for disk data
	mov	dl,es:[dskdata];	header checksum

	cmp	dh,dl;			compare checksum
	jz	zero_seek25;		ok

;
;	error, bad checksum in the header
;
	jmp	zero_softerr;		can retry this one



;
;	have read the header, check if it's the right track
;
zero_seek25:;
	or	al,al;			test if on track zero
	ret;

;
;	possibly retry an error
;
zero_softerr:;
	dec	si;			decrement retry counter
	jnz	zero_seek;		retry

aint_on_zero:;
	mov	al,1;			no, flag an error
	jmp	zero_seek25;		and return (not on track zero)


zero_chk	endp;



;
;	select a drive
;
;	input:		bl = drive to select
;
;	results:	ax, cx destroyed
;
sel_drv		proc;

;
;	drive select
;
	mov	cl,es:[read_pera];
	and	cl,0FFh - f_selbit - f_led0 - f_led1;	led, select off

	xor	al,al;			preset to drive 0 select
	or	bl,bl;			drive 1 ?
	jz	want_drive0;		no, skip
	mov	al,f_selbit;		yes, that led and select drive 1

want_drive0:;
	or	al,cl;
	mov	es:[read_pera],al;	turn the proper select on

;
;	stepper motor enabling
;
	mov	ch,es:[read_perb];
	or	ch,f_s0enab or f_s1enab;stepper motors off
	or	bl,bl;			drive 0 ?
	jz	enable_stepper_0;	yes

	and	ch,NOT f_s1enab;	enable stepper 1
	jmp short enabled;

enable_stepper_0:;
	and	ch,NOT f_s0enab;	enable stepper 0

enabled:;
	mov	es:[read_perb],ch;	enable that stepper

	ret;

sel_drv		endp;






;
;	deselect the drives
;
;	inputs:		none
;
;	results:	ax destroyed
;
desel_drvs	proc;

	or	es:[read_perb],f_s0enab or f_s1enab;	stepper motors off

	and	es:[read_pera],NOT (f_led0 + f_led1);	and led's off

	ret;

desel_drvs	endp;




;
;	set motor speed and wait for motor ready
;
;	input:		ax = speed (timer value)
;			bl = drive
;
;	results:	ax, cx, di destroyed
;
motor_speed	proc;

;
;	turn LED on for this drive
;
	mov	cl,f_led0;		preset to drive 0's LED
	or	bl,bl;			doing drive 0 ?
	jz	light_drive0;		yes, skip

	mov	cl,f_led1;		no, set to drive 1's LED

light_drive0:;
	or	es:[read_pera],cl;	put on the LED

;
;	test if on same speed zone as before (then no speed change)
;
	xor	ah,ah;
	mov	di,ax;			get zone to select
	add	di,di;			times 2 bytes per entry
	mov	di,cs:speed_for_zones[di];	get timer value for zone's speed
	cmp	curspd,di;		at that speed already ?
	jz	at_speed;		yes, don't stop and start motor again
	mov	curspd,di;		no, set the new speed

;
;	set motor's speed
;
	and	es:[read_perb],NOT f_screset;	enable motor timer

	mov	cx,seg motor_time;	timer for motor speed control
	mov	es,cx;
	assume	es:motor_time;

	mov	ch,drive_0_timer;	preset to select drive 0
	or	bl,bl;			drive 0 ?
	jz	drive_A_timer;		yes, skip

	mov	ch,drive_1_timer;	select drive 1

drive_A_timer:;
	mov	byte ptr es:[timer_control],ch;		select drive

	mov	cx,di;					get speed value
	mov	byte ptr es:[bx+drive_0_time],cl;	low order byte of speed
	mov	byte ptr es:[bx+drive_0_time],ch;	high order byte of speed

	mov	cx,seg ioports;		address the floppy I/O registers
	mov	es,cx;
	assume	es:ioports;

	mov	ax,4000;		400 milliseconds and we'll be ready
	call	time;			(no other way to test for ready)

at_speed:;
	ret;

motor_speed	endp;






;
;	turn the motors off on both drives
;
;	inputs:		bl = drive's number
;
;	results:	ax, cx destroyed
;
motor_off	proc;

	and	es:[read_perb],NOT f_screset;	enable motor timer

	mov	cx,seg motor_time;	timer for motor speed control
	mov	es,cx;
	assume	es:motor_time;

	mov	ch,drive_0_off;		preset to select drive 0
	or	bl,bl;			drive 0 ?
	jz	drive_A;		yes, skip

	mov	ch,drive_1_off;		select drive 1

drive_A:;
	mov	byte ptr es:[timer_control],ch;		select drive

	mov	byte ptr es:[bx+drive_0_time],0;	low order byte of speed
	mov	byte ptr es:[bx+drive_0_time],0;	high order byte of speed

	mov	cx,seg ioports;		address the floppy I/O registers
	mov	es,cx;
	assume	es:ioports;

	and	es:[read_pera],NOT (f_led0 + f_led1);	turn the led's off, too
	call	desel_drvs;		and deselect both drives

	mov	curspd,0;		flag that we're at no speed zone

	ret;

motor_off	endp;





;
;	return proper speed index for the track's zone
;
;	inputs:		al = track
;
;	outputs:	ax ,cx = speed zone index
;
get_zon		proc;

	push	bp;
	xor	bp,bp;			scan zone table from 0 to 14

next_entry:;
	cmp	al,cs:last_track_for_zone[bp];
	ja	not_the_one;		not speed for this track, skip

	mov	ax,bp;			return the index of the zone
	mov	cx,bp;

	pop	bp;
	ret;

not_the_one:;
	inc	bp;			try next table element
	cmp	bp,speed_zones;		done ?
	jnz	next_entry;		no, loop

	jmp	err_intern;		yes, internal error (didn't find it)

get_zon		endp;




;
;	look for a change in the index sense (for up to 2 revolutions,
;	which is 500 milliseconds)
;
;	inputs:		dl = mask pattern for this drive's index bit
;			dh = old index value (which we wait for to go away)
;
;	results:	al, cx destroyed
;			ah set to the final index bit value
;			z-flag is zero if no change occurred
;			z-flag non-zero when a new index value occurs
;
index_change	proc;

	mov	cx,500;			maximum time to see the hole, 2 revs
;					(237ms per revolution)
index_loop:;
	mov	ax,10;			minimum sampling rate (1ms)
;					(pulse occurs for 2ms)
	call	time;

	mov	ah,es:[read_perb];	get new index state
	and	ah,dl;			look only at this drive's bit
	cmp	ah,dh;			changed from original ?

	loopz	index_loop;		no, check it again

	ret;				return z-flag, ah as result

index_change	endp;




;
;	reset the floppy disk hardware
;
;	inputs:		none
;
;	results:	all registers destroyed
;
resethdw	proc;

	lea	di,ioports:motor_1;	base of I/O registers

	mov	cx,3;			3 sets to clear
;
;	clear programmed interface adapter registers
;
resetpl:;
	mov	dx,cx;
	mov	cx,14;			14 bytes to clear for each set
	cld;
	xor	al,al;
	rep	stosb;			zero the registers

	mov	al,7fh;			get the interrupt enable register
	stosb;				(must be set to 7F)

	mov	cx,dx;			restore set counter
	add	di,17;			displacement to next set
	loop	resetpl;		until all 3 sets cleared

	ret;

resethdw endp;	



code	ends;


	end;
